#ifndef LLF_RSA_SCHEMES_H
#define LLF_RSA_SCHEMES_H

/*
* All the includes that are needed for code using this module to
* Compile correctly should be #included here.
*/

#include "CE2_public.h"
#include "CE2_HASH.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
* Creation date : Fri Mar 09 16:46:00 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains declaration of low level 
* function that supports PKCS#1 v1.5 and PKCS#1 v2.1 schemes
* which worked with LibTomCrypt.
* 
* \version LLF_RSA_SCHEMES.h#1:csrc:1
* \author Pavel Sasunkevich
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_DX_RSA_SCHEMES_Encrypt
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key data structure of the User;
*  @param hashFunc       [in] - The hash function to be used;
*  @param L              [in] - The label input;
*  @param Llen           [in] - The label length;
*  @param MGF            [in] - The mask generation function. PKCS#1 v2.1 defines MGF1, 
*         so the only value allowed here is CE2_PKCS1_MGF1;
*  @param DataIn_ptr     [in] - A pointer to the data to encrypt;
*  @param DataInSize       [in] - The size, in bytes, of the data to encrypt;
*  @param Output_ptr     [out] - A pointer to the encrypted data;
*  @param PKCS1_ver      [in] - Ver 1.5 or 2.1, according to the functionality required.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  This function implements the Encrypt algorithm as defined in
*  PKCS#1 v2.1 and PKCS#1 v1.5.
*  The actual macro that will be used by the user is:
*   CE2_RSA_OAEP_Encrypt 		- for v2.1;
*   CE2_RSA_PKCS1v15_Encrypt	- for v1.5.
*
* \b
* Algorithm:
*  -# Register prng and hash;
*  -# Encode packet depending on PKCS ver;
*  -# Compute an RSA modular exponentiation depending on PKCS ver.
***************************************************************/
CE2Error_t LLF_DX_RSA_SCHEMES_Encrypt(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr,/* in */
                        CE2_RSA_HASH_OpMode_t hashFunc,	/* in */
                        DxUint8_t      *L,				/* in */
                        DxUint16_t      Llen,			/* in */
                        CE2_PKCS1_MGF_t MGF,			/* in */
                        DxUint8_t       *DataIn_ptr,	/* in */
                        DxUint16_t      DataInSize,		/* in */
                        DxUint8_t       *Output_ptr,	/* out */
                        CE2_PKCS1_version PKCS1_ver);	/* in */

/**
****************************************************************
* Function Name: 
*  LLF_DX_RSA_SCHEMES_Decrypt
*
* Inputs:
*  @param UserPrivKey_ptr [in] - A pointer to the private key data structure;
*  @param hashFunc        [in] - The hash function to be used;
*  @param L               [in] - The label input;
*  @param Llen            [in] - The label length;
*  @param MGF             [in] - The mask generation function. PKCS#1 v2.1 defines MGF1, 
          so the only value allowed here is CE2_PKCS1_MGF1.
*  @param Data_ptr        [in] - A pointer to the data to decrypt;
*  @param DataSize        [in] - The size, in bytes, of the data to decrypt;
*  @param Output_ptr      [out] - Pointer to the decrypted data;
*  @param OutputSize_ptr  [in/out] - The size of the Output_buffer ptr  [bytes];
*  @param PKCS1_ver       [in] - Ver 1.5 or 2.1, according to the functionality required.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  This function implements the Decrypt algorithm as defined
*  in PKCS#1 v2.1 and PKCS#1 v1.5.
*   The actual macro that will be used by the user is:
*   CE2_RSA_OAEP_Decrypt		- for v2.1;
*   CE2_RSA_PKCS1v15_Decrypt	- for v1.5.
*
* \b
* Algorithm:
*  -# Register prng and hash;
*  -# Compute an RSA modular exponentiation depending on PKCS ver;
*  -# Decode packet depending on PKCS ver.
***************************************************************/
CE2Error_t LLF_DX_RSA_SCHEMES_Decrypt(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,/* in */
                        CE2_RSA_HASH_OpMode_t hashFunc,	/* in */
                        DxUint8_t       *L,				/* in */
                        DxUint16_t      Llen,			/* in */
                        CE2_PKCS1_MGF_t MGF,			/* in */
                        DxUint8_t       *Data_ptr,		/* in */
                        DxUint16_t      DataSize,		/* in */
                        DxUint8_t       *Output_ptr,	/* out */
                        DxUint16_t      *OutputSize_ptr,/* in, out */
                        CE2_PKCS1_version PKCS1_ver);	/* in */

/**
****************************************************************
* Function Name: 
*  LLF_DX_RSA_Sign
*
* Inputs:
*  @param UserPrivKey_ptr [in] - A pointer to the private key data
*         structure of the user;
*  @param hashFunc        [in] - The hash function to be used;
*  @param MGF             [in] - The mask generation function;
*  @param SaltLen         [in] - The Length of the Salt buffer;
*  @param DataIn_ptr      [in] - A pointer to the data to sign;
*  @param DataInSize      [in] - The size, in bytes, of the data to sign;
*  @param Output_ptr      [out] - A pointer to the signature;
*  @param OutputSize_ptr  [in/out] - A pointer to the Signature Size value;
*  @param PKCS1_ver       [in] - Ver 1.5 or 2.1, according to the functionality required.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*     - CE2_RSA_INVALID_PRIV_KEY_STRUCT_POINTER_ERROR,
*     - CE2_RSA_PRIV_KEY_VALIDATION_TAG_ERROR,
*     - CE2_RSA_HASH_ILLEGAL_OPERATION_MODE_ERROR,
*     - CE2_RSA_MGF_ILLEGAL_ARG_ERROR,
*     - CE2_RSA_PKCS1_VER_ARG_ERROR,
*     - CE2_RSA_INVALID_MESSAGE_DATA_SIZE,
*     - CE2_RSA_INVALID_OUTPUT_POINTER_ERROR,
*     - CE2_RSA_INVALID_SIGNATURE_BUFFER_SIZE.
*
* \brief \b
* Description:
*  RSA_Sign implements the Signing algorithm as defined
*  in PKCS#1 v1.5 or v2.1, using a single function
*  The actual macros that will be used by the user are:
*   CE2_RSA_PKCS1v15_Sign		- for v1.5;
*   CE2_RSA_PSS_Sign			- for v2.1.
*								- for v2.1 MD5 is not supported, since 
*                 according to the PKCS#1 ver2.1 it is not recommended.
*
* \b
* Algorithm:
*  -# Register prng and hash;
*  -# Calculate message hash;
*  -# Encode packet depending on PKCS ver - get sign of msg hash.
*  -# Compute an RSA modular exponentiation depending on PKCS ver;
***************************************************************/
CE2Error_t LLF_DX_RSA_Sign(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,/* in */
                        CE2_RSA_HASH_OpMode_t hashFunc,	/* in */
                        CE2_PKCS1_MGF_t MGF,			/* in */
                        DxUint16_t    SaltLen,			/* in */
                        DxUint8_t     *DataIn_ptr,		/* in */
                        DxUint32_t     DataInSize,		/* in */
                        DxUint8_t     *Output_ptr,		/* out */
                        DxUint16_t    *OutputSize_ptr,	/* in, out */
                        CE2_PKCS1_version PKCS1_ver);	/* in */

/**
****************************************************************
* Function Name: 
*  LLF_DX_RSA_Verify
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key data structure of the user;
*  @param hashFunc       [in] - The hash function to be used. 
*  @param MGF            [in] - The mask generation function;
*  @param SaltLen        [in] - The Length of the Salt buffer;
*  @param DataIn_ptr     [in] - A pointer to the data whose signature is to be verified;
*  @param DataInSize     [in] - The size, in bytes, of the data whose signature is to be verified;
*  @param Sig_ptr        [in] - A pointer to the signature to be verified;
*  @param PKCS1_ver      [in] - Ver 1.5 or 2.1, according to the functionality required;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_INVALID_SIGNATURE,
*     - CE2_BAD_PARAM,
*     - CE2_OUT_OF_RANGE.
*
* \brief \b
* Description:
*  RSA_Verify implements the RSASSA-PKCS1v15 algorithm
   in a single function, as defined in PKCS#1 v1.5 and v2.1
*  The actual macro that will be used by the users is:
*   CE2_RSA_PKCS1v15_Verify			- for v1.5;
*   CE2_RSA_PSS_Verify				- for v2.1;
*									- for v2.1 MD5 is not supported, since
*					according to the PKCS#1 ver2.1 it is not recommended.
*
* \b
* Algorithm:
*  -# Register prng and hash;
*  -# Calculate message hash;
*  -# Compute an RSA modular exponentiation depending on PKCS ver;
*  -# Decode packet depending on PKCS ver - get verify msg hash.
***************************************************************/
CE2Error_t LLF_DX_RSA_Verify(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr,/* in */
                        CE2_RSA_HASH_OpMode_t hashFunc,	/* in */
                        CE2_PKCS1_MGF_t MGF,		/* in */
                        DxUint16_t      SaltLen,	/* in */
                        DxUint8_t       *DataIn_ptr,/* in */
                        DxUint32_t      DataInSize,	/* in */
                        DxUint8_t       *Sig_ptr,	/* in */
                        CE2_PKCS1_version PKCS1_ver);/* in */

/**
****************************************************************
* Function Name: 
* LLF_DX_RSA_GetDEREncodingLength
*
*  @param hashFunc [in] - RSA hash function
*  @param pTLength [out] - Returns the DER encoding length (tLen), 
*                          as explained in the EMSA-PKCS1-v1_5
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_UTILS_INVALID_POINTER_ERROR
*
* \brief \b 
* Description:
*  Returns the DER encoding length (tLen), as explained in the EMSA-PKCS1-v1_5
*
*  \b 
* Algorithm:
***************************************************************/
CE2Error_t LLF_DX_RSA_GetDEREncodingLength(CE2_RSA_HASH_OpMode_t hashFunc,
                                           DxUint32_t			       *pTLength);


#ifdef __cplusplus
}
#endif

#endif /* LLF_RSA_SCHEMES_H */
